<?php if ( ! defined( 'ABSPATH' ) ) exit;

final class NF_ConvertKit extends NF_ConvertKit_Plugin
{
    /** @var array|NF_ConvertKit_API */
    private $api;

    public function __construct( $version, $file )
    {
        parent::__construct( $version, $file );

        add_action( 'admin_init', array( $this, 'setup_license' ) );

        add_action( 'admin_init', array( $this, 'enqueue_js' ) );

        add_filter( 'ninja_forms_plugin_settings',             array( $this, 'plugin_settings' ), 10, 1 );
        add_filter( 'ninja_forms_plugin_settings_groups',      array( $this, 'plugin_settings_groups' ), 10, 1 );
        add_action( 'ninja_forms_register_actions',            array( $this, 'register_actions' ) );
        add_filter( 'ninja_forms_field_settings_groups',       array( $this, 'register_settings_groups' ) );
        add_filter( 'nf_convertkit_action_subscribe_settings', array( $this, 'action_subscribe_settings' ) );
        add_action( 'wp_ajax_nf_convertkit_clear_api_cache', array( $this, 'clear_api_cache' ) );
    }

    /*
    |--------------------------------------------------------------------------
    | Action & Filter Callback
    |--------------------------------------------------------------------------
    */

    /*
     * Function to setup the license with our updater.
     */
    public function setup_license()
    {
        if ( ! class_exists( 'NF_Extension_Updater' ) ) return;
        new NF_Extension_Updater( 'ConvertKit', $this->version(), 'The WP Ninjas', $this->dir(), 'convertkit' );
    }

    /*
     * Function to register our plugin settings.
     */
    public function plugin_settings( $settings )
    {
        $settings[ 'convertkit' ] = $this->config( 'plugin-settings' );
        return $settings;
    }

    /*
     * Function to register our plugin settings groups.
     */
    public function plugin_settings_groups( $groups )
    {
        $groups[ 'convertkit' ] = array(
            'id' => 'convertkit',
            'label' => __( 'ConvertKit', 'ninja-forms-convertkit' ),
        );
        return $groups;
    }

    /*
     * Function to register our action.
     */
    public function register_actions( $actions )
    {
        $actions[ 'convertkit' ] = new NF_ConvertKit_Actions_Subscribe();
        return $actions;
    }

    /*
     * Function to register our field settings.
     */
    public function register_settings_groups( $groups )
    {
        $groups[ 'tags' ] = array(
            'id' => 'tags',
            'label' => __( 'Tags', 'ninja-forms-convertkit' ),
            'priority' => 200
        );
        $groups[ 'custom_fields' ] = array(
            'id' => 'custom_fields',
            'label' => __( 'Fields', 'ninja-forms-convertkit' ),
            'priority' => 300
        );
        return $groups;
    }

    /*
     * Function to register settings with our action.
     */
    public function action_subscribe_settings( $settings )
    {
        $api_key = Ninja_Forms()->get_setting( 'convertkit_api_key' );
        if( ! $api_key ){
            // If no API Key is found, then replace options with a message.
            return array( $settings[ 'no_api_key' ] );
        }

        // Remove No API Key message.
        unset( $settings[ 'no_api_key' ] );

        $api = NF_ConvertKit()->api( $api_key );

        // Get the list of Forms for this API key.
        $data = $api->get( 'forms' );
        // If we got back any forms...
        // AND If we got back a non-empty array...
        if ( isset( $data[ 'forms' ] ) && is_array( $data[ 'forms' ] ) && ! empty( $data[ 'forms' ] ) ) {
            // For each form...
            foreach ( $data[ 'forms' ] as $form ) {
                // Add it to our list of available settings.
                $settings[ 'form' ][ 'options' ][] = array(
                    'label' => $form[ 'name' ],
                    'value' => $form[ 'id' ]
                );
            }
        }

        // Get the list of Sequences for this API key.
        $data = $api->get( 'courses' );
        // If we got back any courses...
        // AND If we got back a non-empty array...
        if ( isset( $data[ 'courses' ] ) && is_array( $data[ 'courses' ] ) && ! empty( $data[ 'courses' ] ) ) {
            // For each course...
            foreach ( $data[ 'courses' ] as $sequence ) {
                // Add it to our list of availabe settings.
                $settings[ 'sequence' ][ 'options' ][] = array(
                    'label' => $sequence[ 'name' ],
                    'value' => $sequence[ 'id' ]
                );
            }
        }

        // Get the list of Tags for this API key.
        $data = $api->get( 'tags' );
        // For each tag...
        if ( isset( $data[ 'tags' ] ) && is_array( $data[ 'tags' ] ) 
            && ! empty( $data[ 'tags' ] ) ) {
            foreach( $data[ 'tags' ] as $tag ){
                // Register it as a setting.
                $key = 'tags_' . $tag[ 'id' ];
                $settings[ $key ] = array(
                    'name' => $key,
                    'type' => 'toggle',
                    'label' => $tag[ 'name' ],
                    'width' => 'full',
                    'group' => 'tags'
                );
            }
        }

        // Get the list of Custom Fields for this API key.
        $data = $api->get( 'custom_fields' );
        // For each field...
        if ( isset( $data[ 'custom_fields' ] ) && is_array( $data[ 'custom_fields' ] ) 
            && ! empty( $data[ 'custom_fields' ] ) ) {
            foreach( $data[ 'custom_fields' ] as $field ){
                // Register it as a setting.
                $key = 'custom_field_' . $field[ 'key' ];
                $settings[ $key ] = array(
                    'name' => $key,
                    'type' => 'textbox',
                    'label' => $field[ 'label' ],
                    'width' => 'full',
                    'group' => 'custom_fields',
                    'use_merge_tags' => true
                );
            }
        }

        return $settings;
    }

    /**
     * Enqueue js to clear cache from settings
     */
    public function enqueue_js() {
        wp_register_script( 'nf-convertkit-admin', $this->url( 'assets/js/admin.js' ), array( 'jquery' ), $this->version(), true );
        wp_enqueue_script( 'nf-convertkit-admin' );
    }

    /*
     * Function to clear our API cache.
     */
    public function clear_api_cache()
    {
        global $wpdb;
        $sql = "DELETE FROM `$wpdb->options` WHERE `option_name` LIKE ('_transient%_nf_convertkit_get_%')";
        $wpdb->query( $sql );
        wp_die( 1 );
    }

    /*
    |--------------------------------------------------------------------------
    | Internal API
    |--------------------------------------------------------------------------
    */

    /**
     * Returns a singleton for the ConvertKit API Wrapper (instantiates if necessary).
     * @param string $key The API Key for ConvertKit
     * @return NF_ConvertKit_API
     */
    public function api( $key )
    {
        // Singleton check by API Key.
        if( isset( $this->api[ $key ] ) ){
            return $this->api[ $key ];
        }
        // New instance with API Key.
        return $this->api[ $key ] = new NF_ConvertKit_API( $key );
    }
}
