<?php

namespace YOOtheme\Builder\Wordpress\Toolset;

use YOOtheme\Str;

class Helper
{
    /**
     * @param string $type
     * @param string $name
     *
     * @return array
     */
    public static function groups($type, $name)
    {
        if (!class_exists('\Toolset_Field_Group_Post_Factory')) {
            return [];
        }

        if ($type === 'post') {
            $fieldGroups = \Toolset_Field_Group_Post_Factory::get_instance()
                ->get_groups_by_post_type($name);

            $factory = \Toolset_Field_Group_Post_Factory::get_instance();
        } else if ($type === 'term') {
            $fieldGroups = \Toolset_Field_Group_Term_Factory::get_instance()
                ->get_groups_by_taxonomy($name);

            $factory = \Toolset_Field_Group_Term_Factory::get_instance();
        } else if ($type === 'user') {
            $fieldGroups = \Toolset_Field_Group_User_Factory::get_instance()
                ->get_groups_by_roles();
            $fieldGroups = array_merge(...array_values($fieldGroups));

            $factory = \Toolset_Field_Group_User_Factory::get_instance();
        }

        return array_map(function ($fieldGroup) use ($factory) {
            return $factory->load_field_group($fieldGroup->get_slug());
        }, $fieldGroups);
    }

    public static function fields($type, $fieldSlugs, $loadRFG = true)
    {
        if ($type === 'post') {
            $factory = \Toolset_Field_Definition_Factory_Post::get_instance();
        } else if ($type === 'term') {
            $factory = \Toolset_Field_Definition_Factory_Term::get_instance();
        } else if ($type === 'user') {
            $factory = \Toolset_Field_Definition_Factory_User::get_instance();
        } else {
            return [];
        }

        $rfg_service = new \Types_Field_Group_Repeatable_Service();

        $fields = [];
        foreach ($fieldSlugs as $slug) {
            $field_definition = $factory->load_field_definition($slug);

            if ($field_definition && $field_definition->is_managed_by_types()) {
                $fields[] = $field_definition->get_definition_array();
            } else if ($loadRFG) {
                $repeatableGroup = $rfg_service->get_object_from_prefixed_string($slug);
                $groupFieldSlugs = $repeatableGroup->get_field_slugs();

                if ($groupFieldSlugs) {
                    $fields[] = [
                        'name' => $repeatableGroup->get_name(),
                        'slug' => $slug,
                        'postId' => $repeatableGroup->get_id(),
                        'type' => 'rfg',
                        'fieldSlugs' => $groupFieldSlugs,
                    ];
                }
            }
        }

        return $fields;
    }

    public static function loadField($field, array $config)
    {
        if ($field['type'] === 'post') {
            $post = self::getPostType($field['data']['post_reference_type']);

            if (!$post) {
                return;
            }

            $type = Str::camelCase($post->name, true);
            return ['type' => self::isMultiple($field) ? ['listOf' => $type] : $type] + $config;
        }

        if ($field['type'] === 'date') {
            $config = array_merge_recursive(
                $config,
                ['metadata' => ['filters' => ['date']]]
            );
        }

        if (self::isMultiple($field)) {
            return [
                'type' => [
                    'listOf' => $field['type'] !== 'date' ? 'ValueField' : 'ToolsetDateField',
                ],
            ] + $config;
        }

        return $config;
    }

    public static function getFieldValue($fieldInstance, $field)
    {
        $fieldType = $fieldInstance->get_type();
        if (in_array($fieldType, ['checkboxes', 'radio', 'select'])) {
            $options = $fieldInstance->get_options();

            $value = array_map(function ($option) {
                return $option->get_value();
            }, $options);

            // filter unchecked radio, select options
            $value = array_values(array_filter($value));
        } else if (in_array($fieldType, ['checkbox'])) {
            $option = $fieldInstance->get_option();

            $value = [
                $option->get_value(),
            ];
        } else if (in_array($fieldType, ['post'])) {
            return $fieldInstance->get_post();
        } else {
            $value = $fieldInstance->get_value();
        }

        if (self::isMultiple($field)) {
            return array_map(function ($value) {
                return compact('value');
            }, $value);
        }

        return isset($value[0]) ? $value[0] : null;
    }

    public static function getPostType($post_type)
    {
        global $wp_post_types;

        if (empty($wp_post_types[$post_type]->rest_base) || $wp_post_types[$post_type]->name === $wp_post_types[$post_type]->rest_base) {
            return;
        }

        return $wp_post_types[$post_type];
    }

    public static function isMultiple($field)
    {
        if (in_array($field['type'], ['checkboxes', 'relationship'])) {
            return isset($field['data']['options']) && count($field['data']['options']) > 1;
        }

        return isset($field['data']['repetitive']) && $field['data']['repetitive'] === '1';
    }
}
