<?php

namespace YOOtheme\Builder\Wordpress\Toolset\Type;

use YOOtheme\Builder\Source;
use YOOtheme\Builder\Wordpress\Toolset\Helper;
use YOOtheme\Str;

class FieldsType
{
    /**
     * @var string
     */
    protected $type;

    /**
     * @var string
     */
    protected $name;

    /**
     * @var Source
     */
    protected $source;

    /**
     * Constructor.
     *
     * @param string $type
     * @param string $name
     */
    public function __construct($type, $name = '')
    {
        $this->type = $type;
        $this->name = $name;
    }

    public function __invoke(Source $source)
    {
        $this->source = $source;

        $fields = [];
        $resolvers = [];

        // add custom fields
        foreach (Helper::groups($this->type, $this->name) as $group) {
            foreach (Helper::fields($this->type, $group->get_field_slugs()) as $field) {
                $config = [
                    'type' => 'String',
                    'metadata' => [
                        'label' => $field['name'],
                        'group' => $group->get_display_name(),
                    ],
                ];

                $fieldConfig = $field['type'] !== 'rfg' ? Helper::loadField($field, $config) : $this->loadRfgField($field, $config);
                if (!$fieldConfig) {
                    continue;
                }

                $fields[$field['slug']] = $fieldConfig;
                $resolvers[$field['slug']] = function ($item) use ($field) {
                    return $field['type'] !== 'rfg' ? $this->resolveField($field, $item) : $this->resolveRfgField($field, $item);
                };
            }
        }

        // add post relationships
        if ($this->type === 'post') {
            $relationships = toolset_get_relationships([
                'type_constraints' => [
                    'any' => [
                        'type' => $this->name,
                    ],
                    'parent' => [
                        'domain' => 'posts',
                    ],
                    'child' => [
                        'domain' => 'posts',
                    ],
                ],
                'origin' => 'any',
            ]);

            foreach ($relationships as $relationship) {
                $config = [
                    'metadata' => [
                        'label' => $relationship['labels']['singular'],
                        'group' => 'Relationships',
                    ],
                ];

                // determine role in relationship
                $isParent = $relationship['roles']['parent']['types'][0] === $this->name;

                // find type of related items
                $type = $relationship['roles'][$isParent ? 'child' : 'parent']['types'][0];
                $post = Helper::getPostType($type);

                if (!$post) {
                    continue;
                }

                // referenced posts are handled as standard field
                if ($relationship['origin'] === 'post_reference_field') {
                    continue;
                }

                if ($relationship['cardinality']['type'] === 'one-to-one' || ($relationship['cardinality']['type'] === 'one-to-many' && !$isParent)) {
                    $config['type'] = $type = Str::camelCase($post->name, true);
                    $fields[$relationship['slug']] = $config;
                    $resolvers[$relationship['slug']] = function ($item) use ($relationship, $isParent) {
                        $post = toolset_get_related_post($item, $relationship['slug'], $isParent ? 'child' : 'parent');
                        if ($post !== 0) {
                            return get_post($post);
                        }
                    };
                } else {
                    $relationshipName = Str::camelCase(['toolset', $post->name, $relationship['slug']], true);
                    $this->source->addType($relationshipName, RelationshipType::class, $post->name, $relationship);

                    $config['type'] = ['listOf' => $relationshipName];
                    $fields[$relationship['slug']] = $config;

                    $resolvers[$relationship['slug']] = function ($item) use ($relationship, $isParent) {
                        $roles = toolset_get_related_posts($item, $relationship['slug'], [
                            'query_by_role' => $isParent ? 'parent' : 'child',
                            'role_to_return' => [$isParent ? 'child' : 'parent', 'intermediary'],
                        ]);

                        return array_map(function ($role) use ($isParent) {
                            return [
                                'post' => $role[$isParent ? 'child' : 'parent'],
                                'intermediary' => $role['intermediary'],
                            ];
                        }, $roles);
                    };
                }
            }
        }

        return compact('fields', 'resolvers');
    }

    protected function loadRfgField($field, array $config)
    {
        $type = Str::camelCase(['toolset', 'slug', 'group'], true);
        $this->source->addType($type, GroupType::class, $field);

        return ['type' => ['listOf' => $type]] + $config;
    }

    protected function resolveField($field, $item)
    {
        $fieldService = new \Types_Field_Service(false);
        if ($this->type === 'post') {
            $fieldInstance = $fieldService->get_field(new \Types_Field_Gateway_Wordpress_Post(), $field['id'], $item->ID);
        } else if ($this->type === 'term') {
            $fieldInstance = $fieldService->get_field(new \Types_Field_Gateway_Wordpress_Term(), $field['id'], $item->term_id);
        } else if ($this->type === 'user') {
            $fieldInstance = $fieldService->get_field(new \Types_Field_Gateway_Wordpress_User(), $field['id'], $item->ID);
        }

        if (empty($fieldInstance)) {
            return;
        }

        return Helper::getFieldValue($fieldInstance, $field);
    }

    protected function resolveRfgField($field, $item)
    {
        $rfg_service = new \Types_Field_Group_Repeatable_Service();
        $rfg = $rfg_service->get_object_by_id($field['postId'], $item);

        if (!$rfg) {
            return;
        }

        return $rfg->get_posts();
    }
}
