<?php

namespace YOOtheme\Builder\Wordpress\Source\Type;

use function YOOtheme\app;
use YOOtheme\Arr;
use YOOtheme\Builder\Source;
use YOOtheme\Path;
use YOOtheme\Str;
use YOOtheme\View;

class PostType
{
    /**
     * @var \WP_Post_Type
     */
    protected $type;

    /**
     * @var array
     */
    protected $features = [
        'title' => 'title',
        'author' => 'author',
        'editor' => 'content',
        'excerpt' => 'excerpt',
        'thumbnail' => 'featuredImage',
        'comments' => 'commentCount',
    ];

    /**
     * Constructor.
     *
     * @param \WP_Post_Type $type
     */
    public function __construct(\WP_Post_Type $type)
    {
        $this->type = $type;
    }

    /**
     * @param Source $source
     *
     * @return array
     */
    public function __invoke(Source $source)
    {
        $taxonomies = array_column(array_filter(get_object_taxonomies($this->type->name, 'object'), function ($taxonomy) {
            return $taxonomy->public && $taxonomy->show_ui && $taxonomy->show_in_nav_menus;
        }), 'name', 'label');

        $fields = [

            'title' => [
                'type' => 'String',
                'metadata' => [
                    'label' => 'Title',
                    'filters' => ['limit'],
                ],
            ],

            'content' => [
                'type' => 'String',
                'args' => [
                    'show_intro_text' => [
                        'type' => 'Boolean',
                    ],
                ],
                'metadata' => [
                    'label' => 'Content',
                    'arguments' => [
                        'show_intro_text' => [
                            'label' => 'Intro Text',
                            'description' => 'Show or hide the intro text.',
                            'type' => 'checkbox',
                            'default' => true,
                            'text' => 'Show intro text',
                        ],
                    ],
                    'filters' => ['limit'],
                ],
            ],

            'teaser' => [
                'type' => 'String',
                'args' => [
                    'show_excerpt' => [
                        'type' => 'Boolean',
                    ],
                ],
                'metadata' => [
                    'label' => 'Teaser',
                    'arguments' => [
                        'show_excerpt' => [
                            'label' => 'Excerpt',
                            'description' => 'Display the excerpt field if it has content, otherwise the intro text.',
                            'type' => 'checkbox',
                            'default' => true,
                            'text' => 'Prefer excerpt over intro text',
                        ],
                    ],
                    'filters' => ['limit'],
                ],
            ],

            'excerpt' => [
                'type' => 'String',
                'metadata' => [
                    'label' => 'Excerpt',
                    'filters' => ['limit'],
                ],
            ],

            'date' => [
                'type' => 'String',
                'metadata' => [
                    'label' => 'Date',
                    'filters' => ['date'],
                ],
            ],

            'modified' => [
                'type' => 'String',
                'metadata' => [
                    'label' => 'Modified',
                    'filters' => ['date'],
                ],
            ],

            'commentCount' => [
                'type' => 'String',
                'metadata' => [
                    'label' => 'Comment Count',
                ],
            ],

            'metaString' => [
                'type' => 'String',
                'args' => [
                    'format' => [
                        'type' => 'String',
                    ],
                    'separator' => [
                        'type' => 'String',
                    ],
                    'link_style' => [
                        'type' => 'String',
                    ],
                    'show_publish_date' => [
                        'type' => 'Boolean',
                    ],
                    'show_author' => [
                        'type' => 'Boolean',
                    ],
                    'show_comments' => [
                        'type' => 'Boolean',
                    ],
                    'show_taxonomy' => [
                        'type' => 'String',
                    ],
                    'date_format' => [
                        'type' => 'String',
                    ],
                ],
                'metadata' => [
                    'label' => 'Meta',
                    'arguments' => [

                        'format' => [
                            'label' => 'Format',
                            'description' => 'Display the meta text in a sentence or a horizontal list.',
                            'type' => 'select',
                            'default' => 'list',
                            'options' => [
                                'List' => 'list',
                                'Sentence' => 'sentence',
                            ],
                        ],
                        'separator' => [
                            'label' => 'Separator',
                            'description' => 'Set the separator between fields.',
                            'default' => '|',
                            'enable' => 'arguments.format === "list"',
                        ],
                        'link_style' => [
                            'label' => 'Link Style',
                            'description' => 'Set the link style.',
                            'type' => 'select',
                            'default' => '',
                            'options' => [
                                'Default' => '',
                                'Muted' => 'link-muted',
                                'Text' => 'link-text',
                                'Heading' => 'link-heading',
                                'Reset' => 'link-reset',
                            ],
                        ],
                        'show_publish_date' => [
                            'label' => 'Display',
                            'description' => 'Show or hide fields in the meta text.',
                            'type' => 'checkbox',
                            'default' => true,
                            'text' => 'Show date',
                        ],
                        'show_author' => [
                            'type' => 'checkbox',
                            'default' => true,
                            'text' => 'Show author',
                        ],
                        'show_comments' => [
                            'type' => 'checkbox',
                            'default' => true,
                            'text' => 'Show comment count',
                        ],
                        'show_taxonomy' => [
                            'type' => 'select',
                            'default' => $this->type->name === 'post' ? 'category' : '',
                            'show' => (bool) $taxonomies,
                            'options' => [
                                'Hide term list' => '',
                            ] + array_combine(array_map(function ($name) { return "Show {$name}"; }, array_keys($taxonomies)), $taxonomies),
                        ],
                        'date_format' => [
                            'label' => 'Date Format',
                            'description' => 'Select a predefined date format or enter a custom format.',
                            'type' => 'data-list',
                            'default' => '',
                            'options' => [
                                'Aug 6, 1999 (M j, Y)' => 'M j, Y',
                                'August 06, 1999 (F d, Y)' => 'F d, Y',
                                '08/06/1999 (m/d/Y)' => 'm/d/Y',
                                '08.06.1999 (m.d.Y)' => 'm.d.Y',
                                '6 Aug, 1999 (j M, Y)' => 'j M, Y',
                                'Tuesday, Aug 06 (l, M d)' => 'l, M d',
                            ],
                            'enable' => 'arguments.show_publish_date',
                            'attrs' => [
                                'placeholder' => 'Default',
                            ],
                        ],
                    ],
                ],
            ],

            'featuredImage' => [
                'type' => 'Attachment',
                'metadata' => [
                    'label' => 'Featured Image',
                ],
            ],

            'link' => [
                'type' => 'String',
                'metadata' => [
                    'label' => 'Link',
                ],
            ],

            'author' => [
                'type' => 'User',
                'metadata' => [
                    'label' => 'Author',
                ],
            ],

            'field' => [
                'type' => Str::camelCase([$this->type->name, 'Fields'], true),
                'metadata' => [
                    'label' => 'Fields',
                ],
            ],

        ];

        $metadata = [
            'type' => true,
            'label' => $this->type->labels->singular_name,
        ];

        $resolvers = $source->mapResolvers($this);

        foreach ($taxonomies as $label => $name) {
            $fields["{$name}String"] = [
                'type' => 'String',
                'args' => [
                    'separator' => [
                        'type' => 'String',
                    ],
                    'show_link' => [
                        'type' => 'Boolean',
                    ],
                    'link_style' => [
                        'type' => 'String',
                    ],
                ],
                'metadata' => [
                    'label' => $label,
                    'arguments' => [
                        'separator' => [
                            'label' => 'Separator',
                            'description' => 'Set the separator between terms.',
                            'default' => ', ',
                        ],
                        'show_link' => [
                            'label' => 'Link',
                            'type' => 'checkbox',
                            'default' => true,
                            'text' => 'Show link',
                        ],
                        'link_style' => [
                            'label' => 'Link Style',
                            'description' => 'Set the link style.',
                            'type' => 'select',
                            'default' => '',
                            'options' => [
                                'Default' => '',
                                'Muted' => 'link-muted',
                                'Text' => 'link-text',
                                'Heading' => 'link-heading',
                                'Reset' => 'link-reset',
                            ],
                            'enable' => 'arguments.show_link',
                        ],
                    ],
                ],
            ];
            $resolvers["{$name}String"] = function ($item, $args) use ($name) {
                $args += ['separator' => ', ', 'show_link' => true, 'link_style' => ''];
                $before = $args['link_style'] ? "<span class=\"uk-{$args['link_style']}\">" : '';
                $after = $args['link_style'] ? '</span>' : '';

                if ($args['show_link']) {
                    return get_the_term_list($item->ID, $name, $before, $args['separator'], $after) ?: null;
                }

                $terms = get_the_terms($item->ID, $name);
                return $terms ? implode($args['separator'], array_map(function ($term) { return $term->name; }, $terms)) : null;

            };
        }

        // omit unsupported features
        if ($values = array_diff_key($this->features, get_all_post_type_supports($this->type->name))) {
            $fields = Arr::omit($fields, $values);
        }

        return compact('fields', 'resolvers', 'metadata');
    }

    public function title($post)
    {
        return $post->post_title;
    }

    public function content($post, $args)
    {
        $args += ['show_intro_text' => true];

        // Hint: this returns different results depending on the current view (archive vs. single page)
        $content = get_the_content('', !$args['show_intro_text'], $post);
        $content = str_replace("<span id=\"more-{$post->ID}\"></span>", '', $content);
        $content = apply_filters('the_content', $content);
        $content = str_replace(']]>', ']]&gt;', $content);

        return $content;
    }

    public function teaser($post, $args)
    {
        $args += ['show_excerpt' => true];

        if ($args['show_excerpt'] && has_excerpt($post)) {
            return get_the_excerpt($post);
        }

        $extended = get_extended($post->post_content);

        return $extended['main'];
    }

    public function date($post)
    {
        return $post->post_date;
    }

    public function modified($post)
    {
        return $post->post_modified;
    }

    public function commentCount($post)
    {
        return $post->comment_count;
    }

    public function link($post)
    {
        return get_permalink($post);
    }

    public function featuredImage($post)
    {
        return get_post_thumbnail_id($post) ?: '';
    }

    public function author($post)
    {
        return get_userdata($post->post_author);
    }

    public function metaString($post, array $args)
    {
        $args += ['format' => 'list', 'separator' => '|', 'link_style' => '', 'show_publish_date' => true, 'show_author' => true, 'show_comments' => true, 'show_taxonomy' => $this->type->name === 'post' ? 'category' : '', 'date_format' => ''];

        return app(View::class)->render(Path::get('../../templates/meta'), compact('post', 'args'));
    }

    public function field($post)
    {
        return $post;
    }
}
